<?php
/**
 * Plugin Name: Social Share Buttons
 * Plugin URI: https://www.getpageup.com/social-share-buttons
 * Description: 可配置的文章社交分享按钮（支持微信二维码）
 * Version: 1.0.0
 * Author: getpageup
 * Author URI: https://www.getpageup.com/
 * License: GPL2+
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: social-share-buttons
 * Domain Path: /languages
 */


if (!defined('ABSPATH')) exit;

define('SSB_OPTION_KEY', 'ssb_settings');

/*--------------------------------------------------
 插件激活：默认启用
--------------------------------------------------*/
register_activation_hook(__FILE__, function () {
    if (!get_option(SSB_OPTION_KEY)) {
        add_option(SSB_OPTION_KEY, [
            'enabled' => 1
        ]);
    }
});

/*--------------------------------------------------
 后台菜单
--------------------------------------------------*/
add_action('admin_menu', function () {
    add_options_page(
        '社交分享按钮',
        '社交分享按钮',
        'manage_options',
        'ssb-settings',
        'ssb_settings_page'
    );
});

/*--------------------------------------------------
 后台设置页
--------------------------------------------------*/
function ssb_settings_page() {
    $options = get_option(SSB_OPTION_KEY, []);

    // 所有分类
    $categories = get_categories(['hide_empty' => false]);

    // 所有公开文章类型
    $post_types = get_post_types(['public' => true], 'objects');

    // JS 支持的平台（完整）
    $platforms = [
        'weibo'     => 'Weibo',
        'wechat'    => 'WeChat',
        'qq'        => 'QQ',
        'qzone'     => 'QZone',
        'tencent'  => 'Tencent Weibo',
        'douban'   => 'Douban',
        'linkedin' => 'LinkedIn',
        'facebook' => 'Facebook',
        'twitter'  => 'Twitter',
        'google'   => 'Google+',
        'diandian' => 'Diandian',
    ];
    ?>
    <div class="wrap">
        <h1>社交分享按钮设置</h1>

        <form method="post" action="options.php">
            <?php settings_fields('ssb_settings_group'); ?>

            <table class="form-table">

                <!-- 启用 -->
                <tr>
                    <th>是否启用</th>
                    <td>
                        <label>
                            <input type="checkbox"
                                   name="<?php echo SSB_OPTION_KEY; ?>[enabled]"
                                   value="1"
                                <?php checked($options['enabled'] ?? 1, 1); ?>>
                            启用分享按钮（默认开启）
                        </label>
                    </td>
                </tr>

                <!-- 文章类型 -->
                <tr>
                    <th>显示文章类型</th>
                    <td>
                        <?php foreach ($post_types as $type) : ?>
                            <label style="display:block;">
                                <input type="checkbox"
                                       name="<?php echo SSB_OPTION_KEY; ?>[post_types][]"
                                       value="<?php echo esc_attr($type->name); ?>"
                                    <?php
                                    if (!empty($options['post_types']) &&
                                        in_array($type->name, $options['post_types'])) {
                                        echo 'checked';
                                    }
                                    ?>>
                                <?php echo esc_html($type->labels->singular_name); ?>
                                <span style="color:#888;">(<?php echo esc_html($type->name); ?>)</span>
                            </label>
                        <?php endforeach; ?>
                        <p class="description">不勾选则所有文章类型显示</p>
                    </td>
                </tr>

                <!-- 平台 -->
                <tr>
                    <th>显示平台</th>
                    <td>
                        <?php
                        $saved = $options['platforms'] ?? [];
                        foreach ($platforms as $key => $label) :
                        ?>
                            <label style="display:inline-block;margin-right:15px;">
                                <input type="checkbox"
                                       name="<?php echo SSB_OPTION_KEY; ?>[platforms][]"
                                       value="<?php echo esc_attr($key); ?>"
                                    <?php checked(in_array($key, $saved)); ?>>
                                <?php echo esc_html($label); ?>
                            </label>
                        <?php endforeach; ?>
                        <p class="description">不勾选则显示全部平台</p>
                    </td>
                </tr>

                <!-- 分类 -->
                <tr>
                    <th>显示分类（仅 post）</th>
                    <td>
                        <?php foreach ($categories as $cat) : ?>
                            <label style="display:block;">
                                <input type="checkbox"
                                       name="<?php echo SSB_OPTION_KEY; ?>[categories][]"
                                       value="<?php echo $cat->term_id; ?>"
                                    <?php
                                    if (!empty($options['categories']) &&
                                        in_array($cat->term_id, $options['categories'])) {
                                        echo 'checked';
                                    }
                                    ?>>
                                <?php echo esc_html($cat->name); ?>
                            </label>
                        <?php endforeach; ?>
                        <p class="description">不勾选则 post 所有分类显示</p>
                    </td>
                </tr>

            </table>

            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

add_action('admin_init', function () {
    register_setting('ssb_settings_group', SSB_OPTION_KEY);
});

/*--------------------------------------------------
 前端资源
--------------------------------------------------*/
add_action('wp_enqueue_scripts', function () {
    if (!is_singular()) return;

    $options = get_option(SSB_OPTION_KEY);
    if (empty($options['enabled'])) return;

    wp_enqueue_style(
        'ssb-share-css',
        plugin_dir_url(__FILE__) . 'share/css/share.min.css',
        [],
        '1.0'
    );

    wp_enqueue_script(
        'ssb-share-js',
        plugin_dir_url(__FILE__) . 'share/js/social-share.min.js',
        [],
        '1.0',
        true
    );
});

/*--------------------------------------------------
 判断文章类型
--------------------------------------------------*/
function ssb_match_post_type() {
    $options = get_option(SSB_OPTION_KEY);

    if (empty($options['post_types'])) {
        return true;
    }

    return in_array(get_post_type(), $options['post_types']);
}

/*--------------------------------------------------
 判断分类（仅 post）
--------------------------------------------------*/
function ssb_match_category() {
    if (get_post_type() !== 'post') {
        return true;
    }

    $options = get_option(SSB_OPTION_KEY);

    if (empty($options['categories'])) {
        return true;
    }

    $post_cats = wp_get_post_categories(get_the_ID());
    return !empty(array_intersect($post_cats, $options['categories']));
}

/*--------------------------------------------------
 输出分享按钮
--------------------------------------------------*/
function ssb_render_share() {
    $options = get_option(SSB_OPTION_KEY);

    if (empty($options['enabled'])) return '';
    if (!ssb_match_post_type()) return '';
    if (!ssb_match_category()) return '';

    // 如果未选择平台 → 使用全部平台
    $all_platforms = [
        'weibo','wechat','qq','qzone','tencent',
        'douban','linkedin','facebook','twitter','google','diandian'
    ];

    $sites = !empty($options['platforms'])
        ? implode(',', $options['platforms'])
        : implode(',', $all_platforms);

    return sprintf(
        '<div class="social-share" data-title="%s" data-url="%s" data-sites="%s"></div>',
        esc_attr(get_the_title()),
        esc_url(get_permalink()),
        esc_attr($sites)
    );
}

/*--------------------------------------------------
 自动插入内容底部
--------------------------------------------------*/
add_filter('the_content', function ($content) {
    if (is_singular() && in_the_loop() && is_main_query()) {
        $content .= ssb_render_share();
    }
    return $content;
});
